//package globalExamples;

import oracle.olapi.data.source.DataProvider;
import oracle.olapi.data.source.FundamentalMetadataProvider;
import oracle.olapi.data.source.FundamentalMetadataObject;
import oracle.olapi.data.source.NumberSource;
import oracle.olapi.data.source.Source;
import oracle.olapi.data.source.StringSource;

import oracle.olapi.metadata.mdm.MdmCustomObjectFactory;
import oracle.olapi.metadata.mdm.MdmLevelHierarchy;
import oracle.olapi.metadata.mdm.MdmMeasure;
import oracle.olapi.metadata.mdm.MdmMeasureDimension;
import oracle.olapi.metadata.mdm.MdmMetadataProvider;
import oracle.olapi.metadata.mdm.MdmPrimaryDimension;
import oracle.olapi.metadata.mdm.MdmSchema;

import java.util.List;
import java.util.Iterator;

/**
 * Complete code for Example 2-4, Creating a Custom Member of the MdmMeasureDimension,
 * in Chapter 2, Understanding OLAP API Metadata, in the Oracle OLAP Developer's
 * Guide to the OLAP API.
 * This example produces the same measure data as Example 5-4, Creating A Custom
 * Measure, but it includes the short descriptions of the selected dimension members.
 *
 * This program uses the Context10g class, which uses the
 * CursorPrintWriter class.
 *
 * @author Oracle Corporation
 */
public class CreateCustomMeasureDimensionMember
{
  public void run(String [] args)
  {
    // Create a PrintWriter and objects for connecting to the database and
    // getting the metadata.
    Context10g context = new Context10g(args, false);

    context.println("Example 2-4, Creating a Custom Member of the MdmMeasureDimension\n");

    // Get the DataProvider.
    DataProvider dp = context.getDataProvider();

    // Get the measures and the Source objects for the measures.
    MdmMeasure mdmUnitPrice = context.getMdmMeasureByName("UNIT_PRICE");
    NumberSource unitPrice = (NumberSource) mdmUnitPrice.getSource();

    MdmMeasure mdmUnitCost = context.getMdmMeasureByName("UNIT_COST");
    NumberSource unitCost = (NumberSource) mdmUnitCost.getSource();

    // Get the primary dimensions for the measures.
    MdmPrimaryDimension mdmProdDim = context.getMdmPrimaryDimensionByName("PRODUCT");
    MdmPrimaryDimension mdmTimeDim = context.getMdmPrimaryDimensionByName("TIME");

    // Get the default level hierarchies of the dimensions.
    MdmLevelHierarchy mdmProdRollup = (MdmLevelHierarchy)
                                      mdmProdDim.getDefaultHierarchy();
    MdmLevelHierarchy mdmCalendar = (MdmLevelHierarchy)
                                    mdmTimeDim.getDefaultHierarchy();

    // Get the Source objects for the hierarchies.
    StringSource prodRollup = (StringSource) mdmProdRollup.getSource();
    StringSource calendar = (StringSource) mdmCalendar.getSource();

    // Get the MdmMetadataProvider and the root MdmSchema.
    MdmMetadataProvider mdmMetadataProvider = context.getMetadataProvider();
    MdmSchema rootSchema = mdmMetadataProvider.getRootSchema();

    // Get the MdmMeasureDimension of the root MdmSchema, and the Source for it.
    MdmMeasureDimension mdmMeasDim = rootSchema.getMeasureDimension();
    StringSource measDim = (StringSource) mdmMeasDim.getSource();

    // Specify a calculation for the custom measure.
    Source calculation = unitPrice.minus(unitCost);

    // Create custom measure as a custom member of the MdmMeasureDimension.
    MdmMeasure mdmMarkup = mdmMeasDim.createCustomMeasure("MARKUP",
                                                          rootSchema,
                                                          calculation,
                                                          10);
    // Set the short and long value descriptions.
    mdmMarkup.setShortDescription("UNIT_PRICE MINUS UNIT_COST");
    mdmMarkup.setDescription("Unit price minus unit cost.");

    // Produce a Source that specifies the values of the members of the
    // measure dimension.
    Source measDimSel = measDim.selectValues(new String[]
                                                 {mdmMarkup.getValue(),
                                                  mdmUnitCost.getValue(),
                                                  mdmUnitPrice.getValue()});

    // Get the Source objects for the short value description attributes
    // of the Product and Time dimensions.
    Source timeShortDesc =
                    mdmTimeDim.getShortValueDescriptionAttribute().getSource();
    Source prodShortDesc =
                    mdmProdDim.getShortValueDescriptionAttribute().getSource();

    // Select members of the product and time hierarchies.
    StringSource prodSel = prodRollup.selectValue("PRODUCT_ROLLUP::ITEM::13");
    StringSource timeSel = calendar.selectValues(new String[]
                                                {"CALENDAR::MONTH::43",
                                                 "CALENDAR::MONTH::44",
                                                 "CALENDAR::MONTH::45"});

    // Produce a query that specifies the values of the members of the measure
    // dimesion for the specified product and times. Include the short descriptions
    // of the Product and Time dimension members.
    Source result = measDim.extract()
                           .join(measDimSel)
                           .join(timeShortDesc.join(timeSel))
                           .join(prodShortDesc.join(prodSel));

    // Prepare and commit the current Transaction.
    context.commit();

    // Create a Cursor for the result Source and display the values of the Cursor.
    context.displayResult(result);

  }

  public static void main(String[] args)
  {
    new CreateCustomMeasureDimensionMember().run(args);
  }
}

